(function() {
  /**
   * WordPress Batch Job Handler
   *
   * @since 4.8.0
   */
  jQuery(function($) {
    'use strict';
    // Handles batch processing job items.

    // @since 4.8.0
    return window.SV_WP_Job_Batch_Handler = class SV_WP_Job_Batch_Handler {
      // Constructs the class.

      // @since 4.8.0

      // @params [Object] args with properties:
      //     id:    job handler ID, used for naming actions and events
      //     nonce: nonce for AJAX requests
      constructor(args) {
        // Processes a given job ID in batches.

        // @since 4.8.0

        // @param [String] an existing job ID
        // @returns [Promise]
        this.process_job = this.process_job.bind(this);
        this.id = args.id;
        this.process_nonce = args.process_nonce;
        this.cancel_nonce = args.cancel_nonce;
        this.cancelled = false;
      }

      process_job(job_id) {
        return new Promise((resolve, reject) => {
          var data;
          if (this.cancelled === job_id) {
            // halt batch processing if a job is cancelled by user action
            return this.cancel_job(job_id);
          }
          data = {
            action: `${this.id}_process_batch`,
            security: this.process_nonce,
            job_id: job_id
          };
          return $.post(ajaxurl, data).done((response) => {
            if (!(response.success && (response.data != null))) {
              // trigger an error if an error is returned or the job data is missing
              return reject(response);
            }
            if (response.data.status !== 'processing') {
              // we're done if the job is anything but still processing
              return resolve(response);
            }
            // broadcast the job progress
            $(document).trigger(`${this.id}_batch_progress_${response.data.id}`, {
              percentage: response.data.percentage,
              progress: response.data.progress,
              total: response.data.total
            });
            // continue processing until finished
            return resolve(this.process_job(response.data.id));
          }).fail(function(jqXHR, textStatus, error) {
            return reject(error); // TODO: anything more we can do here?
          });
        });
      }

      
        // Cancels a given job.

      // @since 4.8.0

      // @param [String] an existing job ID
      // @returns [Promise]
      cancel_job(job_id) {
        return new Promise((resolve, reject) => {
          var data;
          this.cancelled = false;
          data = {
            action: `${this.id}_cancel_job`,
            security: this.cancel_nonce,
            job_id: job_id
          };
          return $.post(ajaxurl, data).done(function(response) {
            if (!response.success) {
              return reject(response);
            }
            return resolve(response);
          }).fail(function(jqXHR, textStatus, error) {
            return reject(error);
          });
        });
      }

    };
  });

}).call(this);

//# sourceMappingURL=sv-wp-admin-job-batch-handler.min.js.map
